// AxsunOCTControl_LW.h
// Copyright 2025 Excelitas Technologies

/** \file AxsunOCTControl_LW.h
\brief This header file defines types and functions used by both the C and C++ interfaces of the AxsunOCTControl_LW API.

*/


#ifndef AXSUNOCTCONTROLLW_H
#define AXSUNOCTCONTROLLW_H

#include "AxsunCommonEnums.h"		// types used by both AxsunOCTControl_LW and AxsunOCTCapture (e.g. AxErr)

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus

	/*! \brief Connection types (interfaces) supported by AxsunOCTControl_LW. */
	typedef enum
#ifdef __cplusplus
		class
#endif // __cplusplus
		AxConnectionType {
		NONE = 0,
			USB = 1,					/*!< USB 2.0 interface. */
			RS232_PASSTHROUGH = 2,		/*!< via DAQ's Ethernet interface. */
			RS232 = 3,					/*!< RS-232 serial interface. */
			ETHERNET = 4				/*!< TCP/IP Ethernet interface. */
	} AxConnectionType;

	/*! \brief Axsun device types supported by AxsunOCTControl_LW. */
	typedef enum
#ifdef __cplusplus
		class
#endif // __cplusplus
		AxDevType {
		UNDEFINED = 0,
			LASER = 40,			/*!< Axsun Laser Engine Device. */
			CLDAQ = 41,			/*!< Axsun Camera Link DAQ Device. */
			EDAQ = 42,			/*!< Axsun Ethernet/PCIe DAQ Device. */
			AZMYTH_LASER = 43	/*!< Axsun Azmyth Laser Engine Device. */
	} AxDevType;

	/*! \brief Laser TEC states. */
	typedef enum
#ifdef __cplusplus
		class
#endif // __cplusplus
		AxTECState {
		TEC_UNINITIALIZED = 0,			/*!< TEC has not been initialized. */
			WARMING_UP = 1,					/*!< TEC is stabilizing. */
			WAITING_IN_RANGE = 2,			/*!< TEC is stabilizing. */
			READY = 3,						/*!< TEC is ready. */
			ERROR_OFF = 4,					/*!< TEC is in error state. */
			NOT_INSTALLED = 5,				/*!< TEC is not installed. */
			DISABLED = 6,					/*!< TEC has been disabled. */
			ERROR_NEVER_GOT_TO_READY = 16,	/*!< TEC error: never got to ready state. */
			ERROR_WENT_OUT_OF_RANGE = 17,	/*!< TEC error: temperature went out of range. */
			TEC_STATE_UNAVAILABLE = 99
	} AxTECState;

	/*! \brief Laser engine default/power-up behaviors. NOTE: Laser engines may be customized to have alternative behavior with regard to swept laser emission and trigger control via the digital input; contact Axsun technical support for details. */
	typedef enum
#ifdef __cplusplus
		class
#endif // __cplusplus
		AxPowerUpBehavior {
		BEHAVIOR_UNAVAILABLE = 0,
			OFF_AT_POWER_UP = 1,			/*!< Laser disabled at power-up. */
			ON_AT_POWER_UP = 2,				/*!< Laser enabled at power-up. */
			USE_DIGITAL_INPUT = 3,			/*!< Swept laser state depends on digital input (Pin 1 on engine's Auxiliary & Interlocks connector). */
			LINK_TO_SWEPT_LASER = 4,		/*!< Pointer/aiming laser state tracks that of swept laser */
			BEHAVIOR_RFU = 5
	} AxPowerUpBehavior;

	/*! \brief Available data types for Named Data field values. */
	typedef enum
#ifdef __cplusplus
		class
#endif // __cplusplus
		AxNamedDataType {
		UNINIT_TYPE = 0,				/*!< type is uninitialized */
			INTEGER_TYPE = 1,				/*!< 32-bit signed integer type */
			BOOL_TYPE = 2,					/*!< Boolean type */
			SINGLE_TYPE = 3					/*!< Single-precision floating point type */
	} AxNamedDataType;

	typedef uint32_t AxDeviceHandle;

	/** \brief Structure for device information available in connect or disconnect callback functions. */
	struct AxDeviceInfo {
		AxDevType device_type;				/*!< device type */
		AxConnectionType connection_type;	/*!< connection type */
		char serial_number[40];				/*!< device serial number string (nul-terminated) */
		char firmware_version[16];			/*!< device firmware version string (nul-terminated) */
		char FPGA_version[16];				/*!< device FPGA version string, if applicable (nul-terminated) */
		AxDeviceHandle handle;				/*!< device unique handle */
	};
	typedef struct AxDeviceInfo AxDeviceInfo;

	/** \brief Union for Named Data field value. */
#ifndef __cplusplus
#include <stdbool.h>
#endif // !__cplusplus
	union AxNamedDataValue {
		uint32_t u;
		int32_t i;
		bool b;
		float f;
	};
	typedef union AxNamedDataValue AxNamedDataValue;

	/*! \brief Type defintion for a user-provided function to be invoked upon connection or disconnection of a device, when registered using axRegisterConnectCallback(). */
	typedef void(__cdecl* AxConnectCallbackFunctionC_t)(void*);

	/*! \brief Type defintion for a user-provided function to be invoked upon device connection if registered using axRegCallbackConnectOnly() or upon device disconnection if registered using axRegCallbackDisconnectOnly(). */
	typedef void(__cdecl* AxCallbackConnectOrDisconnect_t)(AxDeviceInfo, int32_t, void*);

	/**
	 \brief Gets the AxsunOCTControl_LW library version as numeric elements.
	 \param[out] major Version number major element.
	 \param[out] minor Version number minor element.
	 \param[out] patch Version number patch element.
	 \param[out] build Version number build element.
	 \return NO_AxERROR on success.
	 */
	/*! \cond */ AXSUN_EXPORTS /*! \endcond */
		AxErr __cdecl axLibraryVersion(uint32_t* major, uint32_t* minor, uint32_t* patch, uint32_t* build);

	/**
	 \brief Get a string describing the AxsunOCTControl_LW library version.
	 \return A nul-terminated UTF-8 encoded char string representing the library version.
	 */
	/*! \cond */ AXSUN_EXPORTS /*! \endcond */
		const char* axLWLibVersion();

	/**
	 \brief Get a string describing the time and date that the AxsunOCTControl_LW library was built.
	 \param[out] datetime_str A pointer to a pre-allocated output buffer of characters with size at least 100 bytes into which a nul-terminated date & time string will be copied.
	 \return The number of characters (bytes) written into the datetime_str buffer, not including the trailing nul.
	 \details It is unsafe to pass a datetime_str output buffer allocated with fewer than 100 bytes.
	 \deprecated This function may be removed in future versions. Use the safer alternative axLWBuildDateTime().
	 */
	/*! \cond */ AXSUN_EXPORTS /*! \endcond */
		int32_t __cdecl axGetLWBuildDateTime(char* datetime_str);

	/**
	 \brief Get a string describing the time and date that the AxsunOCTControl_LW library was built.
	 \return A nul-terminated UTF-8 encoded char string representing the build date & time.
	 */
	/*! \cond */ AXSUN_EXPORTS /*! \endcond */
		const char* axLWBuildDateTime();

	/**
	\brief Get the build configuration of the AxsunOCTControl_LW library.
	\return = a bitfield with information about the build configuration options used to compile the loaded library.
	\ Bit 0: = 0 for 'debug' configuration or = 1 for 'release' configuration
	\ Bit 1: (for internal use only)
	\ Bit 2: = 0 for non-verbose mode or = 1 for verbose logging mode
	*/
	/*! \cond */ AXSUN_EXPORTS /*! \endcond */
		int32_t __cdecl axGetLWBuildCfg();

	/**
	 \brief Gets a string which explains an error code in a more verbose fashion.
	 \param[in] errornum An error code number returned from other "ax" functions in this library.
	 \param[out] error_string A char array pre-allocated with at least 512 bytes into which the error explanation will be copied as a nul-terminated string.
	 \return n/a
	 \details axGetErrorExplained() can be called at any time. It is unsafe to pass a error_string output buffer allocated with fewer than 512 bytes.
	 */
	/*! \cond */ AXSUN_EXPORTS /*! \endcond */
		void __cdecl axGetErrorExplained(AxErr errornum, char* error_string);

#ifdef __cplusplus
}		// end extern "C"

/*! \brief Convert AxConnectionType enum into string. */
inline const char* AxConnectTypeString(AxConnectionType connection) {
	switch (connection) {
	case AxConnectionType::USB: return "USB";
	case AxConnectionType::RS232: return "RS-232";
	case AxConnectionType::RS232_PASSTHROUGH: return "via DAQ";
	case AxConnectionType::ETHERNET: return "Ethernet";
	default: return "Unknown";
	}
}

/*! \brief Convert AxDevType enum into string. */
inline const char* AxDevTypeString(AxDevType device) {
	switch (device) {
	case AxDevType::LASER: return "Laser";
	case AxDevType::AZMYTH_LASER: return "Azmyth Laser";
	case AxDevType::EDAQ: return "EDAQ";
	case AxDevType::CLDAQ: return "CLDAQ";
	default: return "Unknown";
	}
}

#endif // end __cplusplus

#endif		/* End header include guard */

