// AxsunOCTControl_LW_C.h
// Copyright 2025 Excelitas Technologies

/** \file AxsunOCTControl_LW_C.h
\brief This header file defines the usage of the %AxsunOCTControl_LW "lightweight" library in a C application.
 
Refer to the Main Page for \link cusagesec C API Usage Guidelines \endlink.

DETAILED RELEASE NOTES

Version 1.0.0
- initial release

Version 1.0.1
- improved Linux OS compatbility
- compatible with libusb v1.0.24.11650 when using libusb-win32/libusb0.sys (v1.2.6) driver on Windows OS
- improved type safety when compiled as __cplusplus (uses "enum class")

Version 1.1.0
- added function axGetElapsedTimers() to query elapsed timer values from laser engine

Version 1.2.0
- axGetLaserEmission() and axSetLaserEmission() have independent query/control of laser emission versus sweep-only mode (e.g. for sweep trigger generation without laser emission)

Version 1.3.0
- moved AxEdgeSource definition from AxsunOCTControl_LW_C.h to AxsunCommonEnums.h
- support for Azymth laser engine
- support for synchronous internal Image_sync generation on FPGA version 6.71.0.42 or newer; see description for axSetImageSyncSource()
- __cdecl calling convention added explicitly to function declarations for Windows targets
- added function axGetLaserErrorState() to query potential error states reported by laser engine firmware

Version 1.3.1
- removed preprocessor logic corresponding to LabVIEW event integration and updated axRegisterLabVIEWCallback() parameter to uint32_t*

Version 1.4.0
- added function axSetFPGAWindowFunction() to upload real & imaginary components of the apodization window function to the DAQ FPGA. This convenience function avoids the need for a user to call axSetFPGADataArray() twice.

Version 1.5.0
- added functions axGetLWBuildDateTime() and axGetLWBuildCfg() to query library details and axGetLibusbVersion() to query loaded libusb dependency version
- added callback registration functions axRegCallbackConnectOnly() and axRegCallbackDisconnectOnly() to separate the behavior of the callback registered via axRegisterConnectCallback() which is invoked for both connection and disconnection events
- axSetImageSyncSource() supports selection of connector interface on exposed front edge of 900 MS/s DAQ board when installed in PCIe configuration

Version 1.6.0
- added function axGetDeviceInfo() to query device type, interface, FW & FPGA versions, and serial number in a single function call
- added functions axSetLaserPowerUp(), axGetLaserPowerUp(), axSetPointerPowerUp(), axGetPointerPowerUp() and enum AxPowerUpBehavior to control/query default behavior for the swept OCT laser and auxiliary pointer/aiming laser when the engine powers-up
- added header-only helper functions for enum -> string conversion in C++
- enabled lazy initialization of Control_LW session without explicit call to axOpenAxsunOCTControl(); this function is now optional.

Version 1.6.1
- improved communication stability for laser engine connected via Ethernet DAQ's serial pass-through interface

Version 1.7.0
- added functions axLWLibVersion() and axLWBuildDateTime()
- added enums and functions to support externally-triggered Burst Record control on FPGA version 6.71.0.71 or newer; see description for axSetBurstRecordSource()
- added enums and functions to support sweep trigger edge configuration when using a bidirectionally-sweeping laser engine and FPGA version 6.71.0.74 or newer; see axSetSwpTrigSrcAndEdge()
- added enums and functions for enhanced configuration of DAQ FPGA window functions and dispersion compensation; see axSetFPGAWindowFunc(), axCalcWindowFunc(), and axCalcWindowFuncToFPGA()
- thread safety: library functions which access connected devices can be called concurrently without explicit serialization by the client application
- added access functions for device firmware's "Named Data" fields; see axGetNamedData(), axSetNamedData(), axClearNamedData(), and axClearAllNamedData()

Version 1.7.1
- added function axPartNumber() to retrieve part number string from device
- removed internal-use-only function declarations from AxsunOCTControl_LW_C.h header file

Version 1.8.0
- corrected typo in description of axGetVDLStatus() arguments
- added function axGetLaserDiagnostics()

Version 2.0.0
- added function axGetDeviceHandle() as well as ability to address connected devices by unique handle (in addition to original device index approach)
- added device unique handle to struct AxDeviceInfo
- improved USB connection reliability on fast Linux OS platforms
- added support for ARM64 processors on Linux OS

Version 2.0.2
- support for 32-bit ARM linux

Version 2.1
- moved some header contents to "AxsunOCTControl_LW.h"
- fix minor regressions related to RS-232 interface
- clarifications in function reference documentation
- added compatibility with devices running firmware v1.13.1
- added functions axResetTEC(), axGetFirmwareErrorExplained(), axGetEDAQDiagnostics(), axGetWavelengthAtSweepTrigger()

Version 2.2
- added function axSetupFPGAResampling() to control FPGA-based resampling functionality on supported DAQ hardware

 */

#ifndef AXSUNOCTCONTROLLW_C_H
#define AXSUNOCTCONTROLLW_C_H

#include "AxsunOCTControl_LW.h"        // types common to the C API and C++ API of AxsunOCTControl_LW

#ifdef __cplusplus
extern "C" {
#endif // __cplusplus
	
/*! \brief Device handle substitutable for "which_device", "which_laser", or "which_DAQ" arguments in AxsunOCTControl_LW C API functions. */
	typedef uint32_t AxDeviceHandle;

/** @name Generic Library Functions
 *  These functions are not directly related to a connected device, but are used for opening & closing the Control_LW session, opening & closing communication interfaces, and other session management.
 */
///@{

/**
 \brief Opens the AxsunOCTControl_LW session.
 \param[in] open_all_interfaces If TRUE the USB and Ethernet interfaces will be started automatically. If FALSE no interfaces will be started.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details See \link cusagesec C API Usage Guidelines \endlink on the Main Page for more information.
 \details If interfaces are not opened at the time of this function call, they can be opened later using axNetworkInterfaceOpen(), axUSBInterfaceOpen(), or axSerialInterfaceOpen(). This allows the registration of connection/disconnection callback function(s) _prior to_ opening interfaces and gives the user individual control over which interface(s) to use.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axOpenAxsunOCTControl(uint32_t open_all_interfaces);

/**
 \brief Closes an AxsunOCTControl_LW session previously opened with axOpenAxsunOCTControl().
 \return NO_AxERROR on success or other AxErr error code on failure. */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axCloseAxsunOCTControl();

/**
 \brief Counts the number of devices successfully connected and enumerated by AxsunOCTControl_LW.
 \return The number of connected devices.
 \details Count includes Laser engine and DAQ boards. A device connected on two interfaces concurrently will be counted twice and thus is not a recommended configuration.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
uint32_t __cdecl axCountConnectedDevices();

/**
 \brief Opens, resets, or closes the USB interface.
 \param[in] interface_status =1 opens the interface or resets an existing open interface, =0 closes the interface.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details When the interface is open, devices present on this interface will be detected, enumerated, and trigger the execution of a callback function (if previously registered). Use this function if the interface was not opened automatically during the initial axOpenAxsunOCTControl() call.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axUSBInterfaceOpen(uint32_t interface_status);

/**
 \brief Opens, resets, or closes the Ethernet network interface.
 \param[in] interface_status =1 opens the interface or resets an existing open interface, =0 closes the interface.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details When the interface is open, devices present on this interface will be detected, enumerated, and trigger the execution of a callback function (if previously registered). Use this function if the interface was not opened automatically during the initial axOpenAxsunOCTControl() call.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axNetworkInterfaceOpen(uint32_t interface_status);

/**
 \brief Opens, resets, or closes a RS-232 serial interface on a given port.
 \param[in] interface_status =1 opens an interface or resets an existing open interface, =0 closes an interface if a `port` is provided, or closes all interfaces if `port` is "" (empty string).
 \param[in] port Nul-terminated char array describing the name of the port (e.g. typically "COM_" on Windows or "/dev/tty.___" on Linux/macOS; UTF8-encoded)
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details When the interface is open, a device present on this interface will be detected, enumerated, and trigger the execution of a callback function (if previously registered).
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSerialInterfaceOpen(uint32_t interface_status, const char* port);

/**
 \brief Gets the version of the libusb library loaded with axUSBInterfaceOpen().
 \param[out] major Version number major element.
 \param[out] minor Version number minor element.
 \param[out] patch Version number patch element.
 \param[out] build Version number build element.
 \return NO_AxERROR on success or HOSTSIDE_USB_ERROR if the USB interface has not yet been opened.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axGetLibusbVersion(uint32_t* major, uint32_t* minor, uint32_t* patch, uint32_t* build);

/**
 \brief Registers a callback function to be executed following a device connection or disconnection event.
 \param[in] callback_function A user-supplied function to be called.
 \param[in] userData An arbitrary structure to be made available inside the callback function (or NULL if unused).
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details The supplied callback function will be executed by a background thread when a new device is detected or when an existing device is disconnected. Within the callback function, access is provided to arbitrary user data passed as a void* parameter to this function.  It is recommended that new designs use axRegCallbackConnectOnly() and axRegCallbackDisconnectOnly() if discrimination between device connection and disconnection events is desired.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axRegisterConnectCallback(AxConnectCallbackFunctionC_t callback_function, void* userData);

/**
 \brief Registers a callback function to be executed following a device connection event.
 \param[in] callback_function A user-supplied function to be called.
 \param[in] userData An arbitrary structure to be made available inside the callback function (or NULL if unused).
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details The supplied callback function will be executed by a background thread when a new device is detected. Within the callback function, access is provided to arbitrary user data passed as a void* parameter to this function.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axRegCallbackConnectOnly(AxCallbackConnectOrDisconnect_t callback_function, void* userData);

/**
 \brief Registers a callback function to be executed following a device disconnection event.
 \param[in] callback_function A user-supplied function to be called.
 \param[in] userData An arbitrary structure to be made available inside the callback function (or NULL if unused).
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details The supplied callback function will be executed by a background thread when an existing device is disconnected. Within the callback function, access is provided to arbitrary user data passed as a void* parameter to this function.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axRegCallbackDisconnectOnly(AxCallbackConnectOrDisconnect_t callback_function, void* userData);

/**
 \brief Registers a LabVIEW user event to be executed following a device connection or disconnection event.
 \param[in] refnumptr A refnum pointer of a LabVIEW user event.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details LabVIEW code can be called directly from this library when a new device is detected or when an existing device is disconnected, similar to a C-based callback function registered using axRegisterConnectCallback(). See the %AxsunOCTControl_LW LabVIEW example project and _extcode.h_ header file (provided by National Instruments) for more details.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axRegisterLabVIEWCallback(uint32_t* refnumptr);

///@}


/** @name Generic Device Functions
 *  These functions access connected devices (either laser engine or DAQ).
 */
 ///@{

/**
 \brief Gets the device unique handle.
 \param[out] handle The unique handle for the device. 
 \param[in] which_device The numeric index of the desired device.
 \details The handle can also be used as the "which_device", "which_laser", or "which_DAQ" argument in all other library calls that address a specific connected device.  If a device disconnects and then reconnects, it will be assigned a new unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
*/
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axGetDeviceHandle(AxDeviceHandle* handle, uint32_t which_device);

/**
 \brief Gets the device firmware version.
 \param[out] major Version number major element.
 \param[out] minor Version number minor element.
 \param[out] patch Version number patch element.
 \param[in] which_device The numeric index of the desired device or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axFirmwareVersion(uint32_t* major, uint32_t* minor, uint32_t* patch, uint32_t which_device);

/**
 \brief Gets the device FPGA version.
 \param[out] major Version number major element.
 \param[out] minor Version number minor element.
 \param[out] patch Version number patch element.
 \param[out] build Version number build element.
 \param[in] which_device The numeric index of the desired device or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Devices which do not have an FPGA will return all version elements = 0.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axFPGAVersion(uint32_t* major, uint32_t* minor, uint32_t* patch, uint32_t* build, uint32_t which_device);

/**
 \brief Gets the device type (e.g. Laser, EDAQ).
 \param[out] device_type The type of device.
 \param[in] which_device The numeric index of the desired device or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axDeviceType(AxDevType* device_type, uint32_t which_device);

/**
 \brief Gets the device serial number string.
 \param[out] serial_number A char array pre-allocated with at least 40 bytes into which the device serial number will be copied as a nul-terminated string.
 \param[in] which_device The numeric index of the desired device or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSerialNumber(char* serial_number, uint32_t which_device);

/**
 \brief Gets the device part number string.
 \param[out] part_number A char array pre-allocated with at least 201 bytes into which the device part number will be copied as a nul-terminated string.
 \param[in] which_device The numeric index of the desired device or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axPartNumber(char* part_number, uint32_t which_device);

/**
 \brief Gets the device connection interface (e.g. USB, Ethernet, RS-232).
 \param[out] connection_type The interface on which the device is connected.
 \param[in] which_device The numeric index of the desired device or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axConnectionType(AxConnectionType* connection_type, uint32_t which_device);

/**
 \brief Gets info about a connected device.
 \param[out] device_info Information about the desired device.
 \param[in] which_device The numeric index of the desired device or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axGetDeviceInfo(AxDeviceInfo* device_info, uint32_t which_device);

/**
 \brief Retrieves a Named Data field from a connected device.
 \param[in] named_index The numeric index of the desired Named Data field to retrieve. Must be on the interval from 0 to 49.
 \param[out] named_type Populated with the type of the value retrieved.
 \param[out] named_value A C union populated with the retrieved Named Data field's value.
 \param[out] field_name Populated with the name of retrieved Named Data field.
 \param[in] which_device The numeric index of the desired device or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Device firmware has storage for 50 Named Data fields (indexed from 0 to 49) which have an associated name string, type (int, bool, or float), and value. Named Data is non-volatile storage available for users to store information according to their needs using axSetNamedData(). If the retrieved named_type is UNINIT_TYPE then a valid value has not been stored at that named_index, otherwise use the named_type argument to interpret the valid member of the retrieved named_value union.  The field_name argument must point to a pre-allocated buffer of at least 61 bytes (60 characters in the name plus a terminating nul character) or undefined behavior will result.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axGetNamedData(uint32_t named_index, AxNamedDataType* named_type, AxNamedDataValue* named_value, char* field_name, uint32_t which_device);

/**
 \brief Store a Named Data field on a connected device.
 \param[in] named_index The numeric index of the desired Named Data field to store. Must be on the interval from 0 to 49.
 \param[in] named_type The type of the Named Data value.
 \param[in] named_value A C union containing the Named Data field's value.
 \param[in] field_name The name of the stored Named Data field (truncated to 60 characters).
 \param[in] which_device The numeric index of the desired device or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Device firmware has storage for 50 Named Data fields (indexed from 0 to 49) which have an associated name string, type (int, bool, or float), and value. Named Data is non-volatile storage available for users to store information according to their needs and can be retrieved with axGetNamedData() or cleared with axClearNamedData().
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetNamedData(uint32_t named_index, AxNamedDataType named_type, AxNamedDataValue named_value, const char* field_name, uint32_t which_device);

/**
 \brief Clears a Named Data field on a connected device.
 \param[in] named_index The numeric index of the desired Named Data field to clear. Must be on the interval from 0 to 49.
 \param[in] which_device The numeric index of the desired device or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axClearNamedData(uint32_t named_index, uint32_t which_device);

/**
 \brief Clears all Named Data fields on a connected device.
 \param[in] which_device The numeric index of the desired device or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axClearAllNamedData(uint32_t which_device);

/**
 \brief Gets an error name associated with a device firmware error code.
 \param[in] FW_errornum An error code number with value between -1 and -999 returned from a device via other "ax" functions in this library. 
 \param[out] error_string A char array pre-allocated with at least 41 bytes into which the error name will be copied as a nul-terminated string.
 \param[in] handle The device's unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Use axGetErrorExplained() to query error information about library AxErr codes outside of the range from -1 to -999.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axGetFirmwareErrorExplained(AxErr FW_errornum, char* error_string, AxDeviceHandle handle);

///@}



/** @name Laser Device Functions
 *  These functions access laser engine devices only.
 */
 ///@{

/**
 \brief Enables or disables swept laser emission.
 \param[in] emission_state =0 disables both laser emission and sweep, =1 enables both laser emission and sweep, =2 enables sweep but disables laser emission.
 \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! 
\cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetLaserEmission(uint32_t emission_state, uint32_t which_laser);

/**
 \brief Gets swept laser emission status.
 \param[out] emitting =0 if both laser emission and sweep are disabled, =1 if both laser emission and sweep are enabled, =2 if laser emission is disabled but sweep is enabled.
 \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */	
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axGetLaserEmission(uint32_t* emitting, uint32_t which_laser);

 /**
 \brief Enables or disables pointer/aiming laser emission.
 \param[in] emission_state =1 enables pointer/aiming laser emission, =0 disables pointer/aiming laser emission.
 \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axSetPointerEmission(uint32_t emission_state, uint32_t which_laser);

 /**
 \brief Gets pointer/aiming laser emission status.
 \param[out] emitting Will be non-zero if pointer/aiming laser is emitting or =0 if pointer/aiming laser is not emitting.
 \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */	
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axGetPointerEmission(uint32_t* emitting, uint32_t which_laser);

 /**
 \brief Sets swept laser emission power-up default behavior.
 \param[in] behavior The desired power-up behavior
 \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Saves setting to non-volatile RAM. Restart engine to observe selected power-upbehavior.
 */	
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axSetLaserPowerUp(AxPowerUpBehavior behavior, uint32_t which_laser);

 /**
\brief Gets swept laser emission power-up default behavior.
 \param[out] behavior The currently configured power-up behavior
 \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
*/	
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	AxErr __cdecl axGetLaserPowerUp(AxPowerUpBehavior* behavior, uint32_t which_laser);

/**
\brief Sets pointer/aiming laser emission power-up default behavior.
 \param[in] behavior The desired power-up behavior
 \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Saves setting to non-volatile RAM. Restart engine to observe selected power-up behavior. 
*/	
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
	AxErr __cdecl axSetPointerPowerUp(AxPowerUpBehavior behavior, uint32_t which_laser);

/**
\brief Gets pointer/aiming laser emission power-up default behavior.
 \param[out] behavior The currently configured power-up behavior
 \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
*/	
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axGetPointerPowerUp(AxPowerUpBehavior* behavior, uint32_t which_laser);

/**
 \brief Sets the electronic K-clock delay.
 \param[in] delay_code A 6-bit code representing the desired delay value.
 \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Discussion of the delay_code value can be found at https://docs.axsun.com (OCT Host page, Clock Delay subsection)
*/
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axSetClockDelay(uint32_t delay_code, uint32_t which_laser);

 /*! \brief Gets the currently configured electronic K-clock delay.
  \param[out] delay_code Will contain a 6-bit code representing the configured delay value.
  \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
  \return NO_AxERROR on success or other AxErr error code on failure.
  \details See details for axSetClockDelay().
 */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axGetClockDelay(uint32_t* delay_code, uint32_t which_laser);

 /**
\brief Selects a pre-programmed laser drive configuration.
\param[in] which_config The desired configuration number (0, 1, 2, or 3).
\param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
\details Laser Engines must be programmed during manufacture to support multiple drive configurations.  Contact Axsun technical support for more information.
*/
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axSetDriveConfiguration(uint32_t which_config, uint32_t which_laser);

 /**
  \brief Gets the currently selected drive configuration.
  \param[out] current_configuration The currently selected drive configuration.
  \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
  \return NO_AxERROR on success or other AxErr error code on failure.
  \details Laser Engines must be programmed during manufacture to support multiple drive configurations.  Contact Axsun technical support for more information.
  */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axGetDriveConfiguration(uint32_t* current_configuration, uint32_t which_laser);

 /**
  \brief Gets elapsed timer values from a laser engine.
  \param[out] cum_emission_sec Number of seconds elapsed with laser in emission-on state (cumulative lifetime).
  \param[out] cum_system_sec Number of seconds elapsed with engine in powered-on state (cumulative lifetime).
  \param[out] uptime_msec Number of milliseconds elapsed in current session (since last power-up).
  \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
  \return NO_AxERROR on success or other AxErr error code on failure.
  */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axGetElapsedTimers(uint32_t* cum_emission_sec, uint32_t* cum_system_sec, uint32_t* uptime_msec, uint32_t which_laser);

 /**
  \brief Gets the current state of a Laser Thermo-Electric Cooler (TEC).
  \param[out] TEC_state Current state of the queried laser TEC as described by the AxsunOCTControl_LW::TECState enum.
  \param[in] which_TEC The index of which TEC to query (1 or 2).
  \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
  \return NO_AxERROR on success or other AxErr error code on failure.
  \details Some Laser Engines support more than one TEC. To query the state of the primary "TEC 1", use `which_TEC` = 1, or for a secondary TEC use `which_TEC` = 2.
  */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axGetTECState(AxTECState* TEC_state, uint32_t which_TEC, uint32_t which_laser);

 /**
 \brief Resets a Laser Thermo-Electric Cooler (TEC).
 \param[in] which_TEC The index of which TEC to reset (1 or 2).
 \param[in] handle The device's unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Some Laser Engines support more than one TEC. To reset the primary "TEC 1", use `which_TEC` = 1, or for a secondary TEC use `which_TEC` = 2.
 */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axResetTEC(uint32_t which_TEC, AxDeviceHandle handle);

 /**
 \brief Gets the current error state of a Laser device.
 \param[out] error_bitfield Current error state of the laser.
 \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details When error_bitfield is equal to zero, there are no errors reported by the device. To decode a non-zero value of error_bitfield, contact Axsun technical support.  Use axGetTECState() to debug TEC-related errors potentially caused by insufficient thermal management.
 */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axGetLaserErrorState(uint32_t* error_bitfield, uint32_t which_laser);

 /**
\brief Gets diagnostic values and their field names from a Laser device.
\param[out] num_of_values Total number of diagnostic value fields returned by the laser (maxiumum of 40 fields).
\param[out] values Pointer to a pre-allocated array of 40 floats into which diagnostic values will be copied.
\param[out] names Pointer to a pre-allocated array of 40 pointers, each representing a pre-allocated array of 41 single-byte characters, into which nul-terminated field name strings will be copied (or pass NULL if field name query isn't needed).
\param[in] which_laser The numeric index of the desired Laser or the device unique handle.
\return NO_AxERROR on success or other AxErr error code on failure.
\details Contact Axsun technical support to interpret laser diagnostic information. Only the first "num_of_values" elements referenced by the "values" and "names" pointer arguments will contain valid data after successfully calling this function.
*/
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
 AxErr __cdecl axGetLaserDiagnostics(uint32_t* num_of_values, float* values, char** names, uint32_t which_laser);

 /**
  \brief Starts the VDL home operation.
  \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
  \return NO_AxERROR on success or other AxErr error code on failure.
  \details This function returns after _requesting_ the homing operation from the VDL controller; it does not wait for the mechanical homing operation to complete before returning.
  */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axHomeVDL(uint32_t which_laser);

 /**
  \brief Stops any VDL operation (e.g. move or home) currently in progress.
  \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
  \return NO_AxERROR on success or other AxErr error code on failure.
  */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axStopVDL(uint32_t which_laser);

 /**
  \brief Move the VDL relative to its current position.
  \param[in] rel_position The distance to move from the current position in mm.
  \param[in] speed The speed at which to move in mm/sec.
  \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
  \return NO_AxERROR on success or other AxErr error code on failure.
  */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axMoveRelVDL(float rel_position, float speed, uint32_t which_laser);

 /**
  \brief Move the VDL to an absolute position.
  \param[in] abs_position The new desired position in mm.
  \param[in] speed The speed at which to move in mm/sec.
  \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
  \return NO_AxERROR on success or other AxErr error code on failure.
  */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axMoveAbsVDL(float abs_position, float speed, uint32_t which_laser);

 /**
  \brief Gets the current status of the VDL.
  \param[out] current_pos Current VDL position (mm).
  \param[out] target_pos Target VDL position (mm).  Will be different than the current position during a move or home operation.
  \param[out] speed Current speed of VDL (mm/sec).
  \param[out] error_from_last_home Accumulated error since last home operation (number of microsteps).
  \param[out] last_move_time Time for last move operation (milliseconds).
  \param[out] state Integer codes to represent VDL state.
										 *	- 1 = idle
										 *	- 2 = moving
										 *	- 3 = seeking home
										 *	- 4 = backing off home
										 *	- 5 = at home
										 *	- 10 = error state
  \param[out] home_switch Non-zero if home (minimum position) limit switch is activated, =0 otherwise.
  \param[out] limit_switch Non-zero if maximum position limit switch is activated, =0 otherwise.
  \param[out] VDL_error Debugging information for VDL errors.
  \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
  \return NO_AxERROR on success or other AxErr error code on failure.
  */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axGetVDLStatus(float* current_pos, float* target_pos, float* speed,
		 int32_t* error_from_last_home, uint32_t* last_move_time,
		 uint8_t* state, uint8_t* home_switch, uint8_t* limit_switch, uint8_t* VDL_error, uint32_t which_laser);


 /**
 \brief Gets the optical wavelength (nm) corresponding to the active sweep trigger edge.
 \param[out] sweep_trig_wavelength_nm The wavelength corresponding to the active sweep trigger edge for the current drive configuration.
 \param[in] which_laser The numeric index of the desired Laser or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
  \details This value is characterized at the time of laser engine manufacture and saved to device memory. It does not represent a dynamically-measured wavelength and therefore may have small errors (~nm) in the returned value relative to the actual sweep trigger wavelength.
 */
 /*! \cond */ AXSUN_EXPORTS /*! \endcond */
	 AxErr __cdecl axGetWavelengthAtSweepTrigger(float* sweep_trig_wavelength_nm, uint32_t which_laser);


///@}


/** @name DAQ Device Functions
 *  These functions access DAQ devices only.
 */
 ///@{

/**
 \brief Control the DAQ operational mode (Live Imaging, Burst Recording, or Imaging Off) when using the Ethernet interface.
 \param[in] number_of_images =0 for Imaging Off (idle), =-1 for Live Imaging (no record), or any positive value between 1 and 32767 to request the desired number of images in a subsequent Burst Record operation.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details This function does NOT control the laser and therefore the laser emission must be enabled/disabled separately using axSetLaserEmission(). To control the DAQ operational mode when using the PCIe interface, call axImagingCntrlPCIe() in the AxsunOCTCapture library.  When number_of_images >0, this function starts the Burst Record operation immediately if the trigger mode is set to SOFTWARE_RECORD_TRIGGER by the axSetBurstRecordSource() function.
*/
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axImagingCntrlEthernet(int16_t number_of_images, uint32_t which_DAQ);

/**
 \brief Select the Image_sync source.
 \param[in] source The desired Image_sync source. Must be either INTERNAL, LVCMOS, LVDS, or INTERNAL_SYNC.
 \param[in] frequency The Image_sync frequency; this parameter is optional and is ignored when source is not INTERNAL or INTERNAL_SYNC.  If the source is INTERNAL, this value is interpreted in Hz and the generated Image_sync is asynchronous with the sweep trigger.  If the source is INTERNAL_SYNC, this value is interpreted in sweep triggers per image and the generated Image_sync is synchronous with the sweep trigger.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Use INTERNAL or INTERNAL_SYNC Image_sync source only for simulation and troubleshooting.  INTERNAL_SYNC requires FPGA version 6.71.0.42 or later.  The source should NOT be configured to use the same external input as axSetBurstRecordSource() configures for the Burst Record trigger source.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetImageSyncSource(AxEdgeSource source, float frequency, uint32_t which_DAQ);

/**
 \brief Select the Burst Record trigger source and behavior.
 \param[in] source The desired Burst Record trigger source.
 \param[in] behavior The desired Burst Record trigger behavior (for externally triggered sources).
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details When source is set to INTERNAL the axImagingCntrlEthernet() function is used to start the Burst Record operation, otherwise the Burst Record is controlled by the external source as configured.  The trigger source should NOT be configured to use the same external input as axSetImageSyncSource() configures for the Image_sync trigger source.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetBurstRecordSource(AxEdgeSource source, AxRcrdTrigBehavior behavior, uint32_t which_DAQ);

/**
 \brief Select the ADC sample clock source.
 \param[in] source The desired sample clock source. Must be either EXTERNAL (Laser k-clock) or INTERNAL (500 MS/s).
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Use INTERNAL sample clock source only for simulation and troubleshooting.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetSampleClockSource(AxEdgeSource source, uint32_t which_DAQ);

/**
 \brief Select the sweep trigger source.
 \param[in] source The desired sweep trigger source. Must be EXTERNAL (from Laser Engine) or INTERNAL (for simulation purposes).
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \deprecated This function may be removed in future versions. Use the alternative axSetSwpTrigSrcAndEdge().
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetSweepTriggerSource(AxEdgeSource source, uint32_t which_DAQ);

/**
 \brief Select the sweep trigger source.
 \param[in] source The desired sweep trigger source. Must be EXTERNAL (from Laser Engine) or INTERNAL (for simulation purposes).
 \param[in] trig_edge The desired sweep trigger edge (FALLING and DUALEDGE modes require compatible laser engine).
 \param[in] trig_freq_Hz The desired INTERNAL sweep trigger frequency in Hz (not applicable for EXTERNAL trigger source).
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetSwpTrigSrcAndEdge(AxEdgeSource source, AxSwpTrgMode trig_edge, uint32_t trig_freq_Hz, uint32_t which_DAQ);

/**
 \brief Select the DAQ's pipeline mode (i.e. bypass mode) and polarization channel configuration.
 \param[in] pipeline_mode The desired pipeline mode.
 \param[in] polarization_mode The desired polarization channel (horizontal, vertical, diverse, or interleaved).
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Insure that A-line subsampling has been configured with SetSubsamplingFactor() \b prior \b to setting a pipeline mode which could saturate the data interface bandwidth if subsampling is insufficient.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetPipelineMode(AxPipelineMode pipeline_mode, AxChannelMode polarization_mode, uint32_t which_DAQ);

/**
 \brief Set the GAIN term during 16- to 8-bit dynamic range compression.
 \param[in] gain Gain must be between 0 and 15.999756.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetEightBitGain(float gain, uint32_t which_DAQ);

/**
 \brief Set the OFFSET term during 16- to 8-bit dynamic range compression.
 \param[in] offset Offset must be between -128.0 and 127.996094.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetEightBitOffset(float offset, uint32_t which_DAQ);

/**
 \brief Set the A-line subsampling factor.
 \param[in] subsampling_factor Subsampling factor must be between 1 and 166.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetSubsamplingFactor(uint8_t subsampling_factor, uint32_t which_DAQ);

/**
 \brief Gets (i.e. reads) the current value in a FPGA register.
 \param[in] regnum The FPGA register number to read.
 \param[out] regval The value read back from the FPGA register.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axGetFPGARegister(const uint32_t regnum, uint16_t* regval, uint32_t which_DAQ);

/**
 \brief Gets (i.e. reads) current values in a range of FPGA registers.
 \param[in] start_regnum The first FPGA register to read in the range.
 \param[in] end_regnum The last FPGA register to read in the range; must be greater than or equal to `start_regnum`.
 \param[out] regvals Pointer to a pre-allocated array to hold the 16-bit register values read back from the requested range of FPGA registers.
 \param[in] bytes_allocated The number of bytes pre-allocated by the user in the array pointed to by `regvals`.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details The size of the pre-allocated memory pointed to by `regvals` must be 2 bytes per register in the included range:  `bytes_allocated` should equal `2 * (end_regnum - start_regnum + 1)`.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axGetFPGARegisterRange(const uint32_t start_regnum, const uint32_t end_regnum, uint16_t* regvals, const uint32_t bytes_allocated, uint32_t which_DAQ);


/**
 \brief Gets (i.e. reads) the programmed power-on default FPGA register configuration script as register number-value pairs.
 \param[in] elements_allocated Indicates the size (<b>in number of U16 elements, NOT bytes</b>) of the pre-allocated `regnums` and `regvals` arrays.
 \param[out] regnums Pointer to a pre-allocated array to hold the 16-bit register numbers read back from the device.
 \param[out] regvals Pointer to a pre-allocated array to hold the 16-bit register values read back from the device.
 \param[out] elements_returned The number of register number-value pairs read back from the device.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details `regnums` and `regvals` are arrays which must be pre-allocated by the user. `elements_returned` will never be higher than `elements_allocated`, however if `elements_returned == elements_allocated`, there may be more register number-value pairs stored on the device which were _not_ returned (indicating `elements_allocated` was too small to hold the full FPGA register configuration script).
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axGetFPGARegisterDefaults(uint32_t elements_allocated, uint16_t* regnums, uint16_t* regvals, uint32_t* elements_returned, uint32_t which_DAQ);


/**
 \brief Sets (i.e. writes) a power-on default FPGA register configuration script, overwriting the existing script.
 \param[in] regnums Pointer to an array containing 16-bit register numbers.
 \param[in] regvals Pointer to an array containing 16-bit register values.
 \param[in] count The number of register number-value pairs to be written to the device.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details `regnums` points to an array containing `count` register numbers and `regvals` points to an equal-sized array of corresponding register default values.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetFPGARegisterDefaults(const uint16_t* regnums, const uint16_t* regvals, uint32_t count, uint32_t which_DAQ);

/**
 \brief Sets (i.e. writes) a FPGA register with a single value.
 \param[in] regnum The FPGA register number in which to write.
 \param[in] regval The value to be written.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetFPGARegister(uint32_t regnum, uint16_t regval, uint32_t which_DAQ);

/**
 \brief Sets (i.e. writes) FPGA register bits based on a bitmask.
 \param[in] regnum The FPGA register number in which to write.
 \param[in] regval The value to be written, subject to the provided bitmask.
 \param[in] bitmask A mask indicating which bits to write (1) and which to leave unaffected (0).
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetFPGARegisterMasked(uint32_t regnum, uint16_t regval, uint16_t bitmask, uint32_t which_DAQ);

/**
 \brief Sets or Clears a single bit in a FPGA register.
 \param[in] regnum The FPGA register number in which to set or clear a bit.
 \param[in] which_bit The bit number in the interval [0..15] to set or clear.
 \param[in] value =1 to Set the bit, =0 to Clear the bit.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetFPGARegisterSingleBit(uint32_t regnum, uint32_t which_bit, uint32_t value, uint32_t which_DAQ);

/**
\brief Sets (i.e. writes) the low or high byte of an FPGA register, leaving the other byte unchanged.
\param[in] regnum The FPGA register number in which to write.
\param[in] which_byte Set to = 0 for low byte, = 1 for high byte.
\param[in] value The value to be written (must be in range of 0 to 255).
\param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
\return NO_AxERROR on success or other AxErr error code on failure.
*/
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetFPGARegisterSingleByte(uint32_t regnum, uint32_t which_byte, uint32_t value, uint32_t which_DAQ);

/**
\brief Sets (i.e. writes) one nibble (4 bits) of an FPGA register, leaving the other 12 bits unchanged.
\param[in] regnum The FPGA register number in which to write.
\param[in] which_nibble Set to = 0 for lowest nibble, ... , = 3 for highest nibble.
\param[in] value The value to be written (must be in range of 0 to 15).
\param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
\return NO_AxERROR on success or other AxErr error code on failure.
*/
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetFPGARegisterSingleNibble(uint32_t regnum, uint32_t which_nibble, uint32_t value, uint32_t which_DAQ);

/**
 \brief Sets (i.e. writes) a FPGA register with an array of multiple values.
 \param[in] regnum The FPGA register number in which to write.
 \param[in] data_array Pointer to an array containing 16-bit values to be written.
 \param[in] number_of_words The number of 16-bit register values pointed to by `data_array`.
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetFPGADataArray(uint32_t regnum, const uint16_t* data_array, uint32_t number_of_words, uint32_t which_DAQ);

/**
 \brief Sets (i.e. writes) an apodization window function to the FPGA.
 \param[in] channel The acquisition channels to which the window function is written.
 \param[in] window_real Pointer to an array containing the real component of the window function.
 \param[in] window_imag Pointer to an array containing the imaginary component of the window function.
 \param[in] window_length The length of the window function (i.e. the number of elements in each array describing its real and imaginary components).
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details A selected AxChannelMode of SUM_CHANNELS or INTERLEAVE_CHANNELS will load the provided window function into both Ch1 and Ch2 (call this function twice if each channel needs to have a differnet window function).  The values in the window_real and window_imag arrays must be on the interval from negative unity to unity [-1.0, 1.0] otherwise clipping will occur and likely result in signal distortion.  If window_length is less-than the appropriate 2^n FFT size (e.g. 2048 samples), the loaded window function will be zero-padded accordingly.
  \deprecated This function may be removed in future versions. Use the alternative axSetFPGAWindowFunc().
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetFPGAWindowFunction(AxChannelMode channel, const float* window_real, const float* window_imag, uint32_t window_length, uint32_t which_DAQ);

/**
 \brief Sets (i.e. writes) an apodization window function to the FPGA.
 \param[in] channel The acquisition channels to which the written window function applies.
 \param[in] trig_edge The sweep trigger edge to which the written window function applies (FALLING and DUALEDGE modes require compatible laser engine).
 \param[in] window_real Pointer to an array containing the real component of the window function.
 \param[in] window_imag Pointer to an array containing the imaginary component of the window function.
 \param[in] window_length The length of the window function (i.e. the number of elements in each array describing its real and imaginary components).
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details A selected AxChannelMode of SUM_CHANNELS or INTERLEAVE_CHANNELS will load the provided window function into both Ch1 and Ch2 (call this function twice if each channel needs to have a differnet window function).  The values in the window_real and window_imag arrays must be on the interval from negative unity to unity [-1.0, 1.0] otherwise clipping will occur and likely result in signal distortion.  If window_length is less-than the appropriate 2^n FFT size (e.g. 2048 samples), the loaded window function will be zero-padded accordingly.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetFPGAWindowFunc(AxChannelMode channel, AxSwpTrgMode trig_edge, const float* window_real, const float* window_imag, uint32_t window_length, uint32_t which_DAQ);

/**
 \brief Calculates a window function according to parameterized description and returns generated values to the caller.
 \param[in] params Parameters describing the complex-valued window function to be generated.
 \param[out] window_real_out Pointer to pre-allocated storage for receiving the real component of the generated window function.
 \param[out] window_imag_out Pointer to pre-allocated storage for receiving the imaginary component of the generated window function.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Calling this function without pre-allocating storage for 'params.padded_width' float-sized elements for each of the window_real and the window_imag outputs is unsafe and will result in undefined behavior. This function does NOT write the generated window function to a connected DAQ FPGA; it only returns the generated window function to the caller. To write a window function to a connected DAQ FPGA, use axSetFPGAWindowFunction() to upload value arrays (such as those generated by this function) or use axCalcWindowFuncToFPGA() to write a window function based on a parameterized description in window_func_t.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axCalcWindowFunc(window_func_t params, float* window_real_out, float* window_imag_out);

/**
 \brief Calculates a window function according to parameterized description and writes it to the FPGA.
 \param[in] channel The acquisition channels to which the written window function applies.
 \param[in] trig_edge The sweep trigger edge to which the written window function applies (FALLING and DUALEDGE modes require compatible laser engine).
 \param[in] params Parameters describing the complex-valued window function to generate and write to the FPGA.
 \param[in] shape Pointer to an array of values used for arbitrary spectral shaping (optional).
 \param[out] window_real_out Pointer to pre-allocated storage for receiving the real component of the generated window function (optional).
 \param[out] window_imag_out Pointer to pre-allocated storage for receiving the imaginary component of the generated window function (optional).
 \param[in] which_DAQ The numeric index of the desired DAQ or the device unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details  After calculating the requested window function with the number of points determined by 'params.window_width', the array subsequently written to the DAQ FPGA is zero-padded to 'params.padded_width' and then rotated right by 'params.right_shift' points.  For example if params.window_width = 1600, this function will add an additional 448 zeros after the requested 1600 points in order to generate a window function appropriate for the subsequent 2048-point FFT computation.  Those 2048 points can be rotated right by using a non-zero value of 'params.right_shift'; this moves the center of the requested window function higher than sample number 800.  The shape pointer argument must be NULL to disable spectral shaping, or otherwise refer to an array of float values with length params.window_width which will be multiplied by the requested window function magnitude. The values in the shape array must be on the interval from zero to unity [0.0, 1.0] otherwise signal distortion from numeric saturation may occur. If non-NULL pointer values are passed in window_real_out and window_imag_out, these pointers must each refer to pre-allocated storage of 'params.padded_width' float-sized elements.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axCalcWindowFuncToFPGA(AxChannelMode channel, AxSwpTrgMode trig_edge, window_func_t params, const float* shape, float* window_real_out, float* window_imag_out, uint32_t which_DAQ);

/**
 \brief Queries some diagnostic values from an Ethernet/PCIe DAQ.
 \param[out] diagnostics Structure of diagnostic values returned from DAQ.
 \param[in] handle The device's unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details Diagnostic values for interpretation by Axsun Technical Support.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axGetEDAQDiagnostics(edaq_diags_t* diagnostics, AxDeviceHandle handle);

/**
 \brief Setup FPGA-based resampling.
 \param[out] param Selects target parameter of the quantity passed via the 'int_val' and/or 'float_val' arguments.
 \param[out] int_val An integer value for the parameter targeted according to the 'param' argument.
 \param[out] float_val A floating point value for the parameter targeted according to the 'param' argument.
 \param[in] handle The device's unique handle.
 \return NO_AxERROR on success or other AxErr error code on failure.
 \details This function sets integer or floating point parameter values associated with FPGA-based resampling on supported DAQ hardware.  Refer to the documentation for the AxResampSetup enum for more details.
 */
/*! \cond */ AXSUN_EXPORTS /*! \endcond */
AxErr __cdecl axSetupFPGAResampling(AxResampSetup param, int32_t int_val, float float_val, AxDeviceHandle handle);

///@}

#ifdef __cplusplus
}		// end extern "C"
#endif

#endif // end header include guard

